import { supabaseAdmin } from "@/lib/supabase/admin";

export async function routeApplication(applicationId: string, topN: number) {
  const admin = supabaseAdmin();

  // Load application
  const { data: app, error: appErr } = await admin
    .from("applications")
    .select("id, loan_product_id, amount_requested, term_requested, status")
    .eq("id", applicationId)
    .single();

  if (appErr) throw new Error(appErr.message);
  if (!app) throw new Error("Application not found");

  // Find eligible issuer products (hard filters)
  const { data: matches, error: matchErr } = await admin
    .from("issuer_products_view")
    .select("issuer_id, issuer_product_id")
    .eq("loan_product_id", app.loan_product_id)
    .lte("min_amount", app.amount_requested)
    .gte("max_amount", app.amount_requested)
    .lte("min_term", app.term_requested)
    .gte("max_term", app.term_requested)
    .eq("issuer_status", "approved")
    .eq("active", true);

  if (matchErr) throw new Error(matchErr.message);

  const uniqueIssuerIds = Array.from(new Set((matches ?? []).map(m => m.issuer_id))).slice(0, topN);

  // Create routing rows (idempotent-ish)
  for (const issuerId of uniqueIssuerIds) {
    await admin.from("application_routing").upsert({
      application_id: applicationId,
      issuer_id: issuerId,
      status: "sent",
      sent_at: new Date().toISOString(),
      last_action_at: new Date().toISOString(),
    }, { onConflict: "application_id,issuer_id" });
  }

  // Update app status
  await admin.from("applications").update({ status: "routed", submitted_at: new Date().toISOString() }).eq("id", applicationId);

  return { routedTo: uniqueIssuerIds.length };
}
